﻿using System;
using System.Runtime.InteropServices;
using System.Threading;

// @Name: ct_single_type_pulse_width.cs
// @Author: Smacq
// @Copyright: Smacq
// @Date: 2024/12/25
// @Brief:
// The Ct functions of USB - 3300 are used for counter acquisition to count rising edges.
// This code acquires 1 data point from CT0 - CT2 channels and calculates single - type pulse width 100 times.
// Only USB - 3300 supports counter functions, Single -> CTxGate.
// You can refer to the user manual for more information about functions or device details.
//
// Execution result when only connecting USB for USB - 3300:
// 0
//...
// 0
// 0 _ct_channel:     ct_remain_buffer: 0     ct data: 0
// pulse_width: 0.0 (s)
//
// 1 _ct_channel:     ct_remain_buffer: 4     ct data: 0
// pulse_width: 0.0 (s)
//
// 2 _ct_channel:     ct_remain_buffer: 6     ct data: 0
// pulse_width: 0.0 (s)
//...
// 0 _ct_channel:     ct_remain_buffer: 0     ct data: 0
// pulse_width: 0.0 (s)
//
// 1 _ct_channel:     ct_remain_buffer: 4     ct data: 0
// pulse_width: 0.0 (s)
//
// 2 _ct_channel:     ct_remain_buffer: 6     ct data: 0
// pulse_width: 0.0 (s)
// 0
// 0
// 0
// 0
// 0
class Program
{
    static void Main(string[] args)
    {
        // Init Ct_data_array
        // Usually the array size need >= acquisition data points size.
        // Data type is unsigned int, it is not modifiable.
        // The data points saved to ct_data_array arrange from get counter channel
        uint[] ct_data_array = new uint[1];

        // Device index starts from zero
        int DevIndex = 0;

        // Counter run mode
        byte CtRunMode = 1;

        // Counter Sample mode
        byte CtSampleMode = 1;

        // Ct Sampling period
        uint CtSampleRate = 1000000000;

        // Acquisition amount of data
        uint Num = 1;

        // Counter trig source
        byte CtTrigSource = 0;

        // Counter acquisition clock source
        byte CtConvSource = 0;

        // Counter channel
        byte ct_channel;

        // measure pulse type: 1=Positive/0=negative
        byte StartEdge = 1;

        // Internal Src signal period
        uint CtInternalSrcPeriod = 10;

        // Timeout unit ms
        int Timeout = 1000;

        int error_code;

        // Open specific device and print return value
        error_code = USB3000DLL.USB3OpenDevice(DevIndex);
        Console.WriteLine(error_code);

        // Set Ct0 - Ct2
        for (ct_channel = 0; ct_channel < 3; ct_channel++)
        {
            // Set counter run mode
            error_code = USB3000DLL.SetCtRunMode(DevIndex, ct_channel, CtRunMode);
            Console.WriteLine(error_code);
            // Set counter sample mode
            error_code = USB3000DLL.SetCtSampleMode(DevIndex, ct_channel, CtSampleMode);
            Console.WriteLine(error_code);
            // Set sample rate
            error_code = USB3000DLL.SetCtSampleRate(DevIndex, ct_channel, CtSampleRate);
            Console.WriteLine(error_code);
            // Set counter trig source
            error_code = USB3000DLL.SetCtTrigSource(DevIndex, ct_channel, CtTrigSource);
            Console.WriteLine(error_code);
            // Set counter clock source
            error_code = USB3000DLL.SetCtConvSource(DevIndex, ct_channel, CtConvSource);
            Console.WriteLine(error_code);
            // Set counter internal Period
            error_code = USB3000DLL.SetCtInternalSrcPeriod(DevIndex, ct_channel, CtInternalSrcPeriod);
            Console.WriteLine(error_code);
            // Set SetCtPulseWidthStartEdge
            error_code = USB3000DLL.SetCtPulseWidthStartEdge(DevIndex, ct_channel, StartEdge);
            Console.WriteLine(error_code);
            // clear counter FIFO
            error_code = USB3000DLL.SetCtClrFifo(DevIndex, ct_channel);
            Console.WriteLine(error_code);
            // Enable soft Trig, start acquisition counter
            error_code = USB3000DLL.SetCtSoftTrig(DevIndex, ct_channel);
            Console.WriteLine(error_code);
        }

        // Continuous execution USB3GetCt() Ct0 - CT2 of 100 times
        for (int i = 0; i < 100; i++)
        {
            for (ct_channel = 0; ct_channel < 3; ct_channel++)
            {
                ulong ct_remain_buffer = (ulong)USB3000DLL.USB3GetCt(DevIndex, ct_channel, Num, ct_data_array, Timeout);
                Console.WriteLine($"{ct_channel} _ct_channel:     ct_remain_buffer: {ct_remain_buffer}     ct data: {ct_data_array[0]}");
                // Calculate pulse width
                double pulse_width = (double)ct_data_array[0] * CtInternalSrcPeriod / 1000000000.0;
                Console.WriteLine($"pulse_width: {pulse_width} (s)");
            }
        }

        // Disable all trig, clear flag
        error_code = USB3000DLL.SetUSB3ClrTrigger(DevIndex);
        Console.WriteLine(error_code);

        // Clear CT FIFO
        for (ct_channel = 0; ct_channel < 3; ct_channel++)
        {
            error_code = USB3000DLL.SetCtClrFifo(DevIndex, ct_channel);
            Console.WriteLine(error_code);
        }

        // Close Device
        error_code = USB3000DLL.USB3CloseDevice(DevIndex);
        Console.WriteLine(error_code);
    }
    class USB3000DLL
    {
        [DllImport("USB3000.dll")]
        public static extern int FindUSB3DAQ();
        [DllImport("USB3000.dll")]
        public static extern int USB3OpenDevice(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int USB3CloseDevice(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int USB3ReadDevcieSN(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int USB3ReadDevcieModel(int DevIndex);

        //--------------------------------------------------------------------------
        // Ananlog Input Configuration

        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiSampleRate(int DevIndex, uint SamplePeriod);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiSampleMode(int DevIndex, byte AiSampleMode);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiConnectType(int DevIndex, byte AiConnectType);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiRange(int DevIndex, byte Chan, float AiRange);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiChanSel(int DevIndex, byte Chan, byte Sel);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiTrigSource(int DevIndex, byte AiTrigSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiConvSource(int DevIndex, byte AiConvSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiPreTrigPoints(int DevIndex, uint AiPreTrigPoints);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiOneShotPoints(int DevIndex, uint AiOneShotPoints);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrAiFifo(int DevIndex);

        //--------------------------------------------------------------------------
        // Digital I/O Configuration

        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DiSampleRate(int DevIndex, uint SamplePeriod);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DiSampleMode(int DevIndex, byte DiSampleMode);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DiTrigSource(int DevIndex, byte DiTrigSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DiConvSource(int DevIndex, byte DiConvSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DiPreTrigPoints(int DevIndex, uint DiPreTrigPoints);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DiOneShotPoints(int DevIndex, uint DiOneShotPoints);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrDiFifo(int DevIndex);

        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoSampleRate(int DevIndex, uint SamplePeriod);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoSampleMode(int DevIndex, byte DoSampleMode);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoTrigSource(int DevIndex, byte DoTrigSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoConvSource(int DevIndex, byte DoConvSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoCycle(int DevIndex, uint DoCycle);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoDataFifo(int DevIndex, uint[] Value, uint Len);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrDoFifo(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoWaveCtrl(int DevIndex, uint Chan);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoImmediately(int DevIndex, uint Chan, uint Value);

        //--------------------------------------------------------------------------
        // Ananlog Output Configuration

        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoSampleRate(int DevIndex, byte Chan, uint SamplePeriod);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoSampleMode(int DevIndex, byte Chan, byte AoSampleMode);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoTrigSource(int DevIndex, byte Chan, byte AoTrigSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoConvSource(int DevIndex, byte Chan, byte AoConvSource);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoCycle(int DevIndex, byte Chan, byte AoCycle);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoDataFifo(int DevIndex, byte Chan, float[] Voltage, uint Len);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrAoFifo(int DevIndex, byte Chan);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoSync(int DevIndex, byte Chans);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoImmediately(int DevIndex, byte Chan, float Voltage);

        //--------------------------------------------------------------------------
        // Timer/Counter Configuration
        [DllImport("USB3000.dll")]
        public static extern int SetCtRunMode(int DevIndex, byte Chan, byte CtRunMode);
        [DllImport("USB3000.dll")]
        public static extern int SetCtSampleMode(int DevIndex, byte Chan, byte CtSampleMode);
        [DllImport("USB3000.dll")]
        public static extern int SetCtSampleRate(int DevIndex, byte Chan, uint SamplePeriod);
        [DllImport("USB3000.dll")]
        public static extern int SetCtTrigSource(int DevIndex, byte Chan, byte CtTrigSource);
        [DllImport("USB3000.dll")]
        public static extern int SetCtConvSource(int DevIndex, byte Chan, byte CtConvSource);
        [DllImport("USB3000.dll")]
        public static extern int SetCtCountEdge(int DevIndex, byte Chan, byte CtCountEdge);
        [DllImport("USB3000.dll")]
        public static extern int SetCtValue(int DevIndex, byte Chan, uint Value);
        [DllImport("USB3000.dll")]
        public static extern int SetCtOverflowValueEdgeCount(int DevIndex, byte Chan, uint OverflowValue);
        [DllImport("USB3000.dll")]
        public static extern int SetCtFrontPartValueEdgeCount(int DevIndex, byte Chan, uint FrontPartValue);
        [DllImport("USB3000.dll")]
        public static extern int SetCtReloadValueEdgeCount(int DevIndex, byte Chan, uint ReloadValue);
        [DllImport("USB3000.dll")]
        public static extern int SetCtDirEdgeCount(int DevIndex, byte Chan, byte Dir);
        [DllImport("USB3000.dll")]
        public static extern int SetCtPulseWidthStartEdge(int DevIndex, byte Chan, byte StartEdge);
        [DllImport("USB3000.dll")]
        public static extern int SetCtInternalSrcPeriod(int DevIndex, byte Chan, uint Period);
        [DllImport("USB3000.dll")]
        public static extern int SetCtSrcSel(int DevIndex, byte Chan, byte SrcSel);
        [DllImport("USB3000.dll")]
        public static extern int SetCtEncodeMode(int DevIndex, byte Chan, byte EncodeMode);
        [DllImport("USB3000.dll")]
        public static extern int SetCtZPhase(int DevIndex, byte Chan, byte ALev, byte BLev);
        [DllImport("USB3000.dll")]
        public static extern int SetCtZValue(int DevIndex, byte Chan, uint Value);
        [DllImport("USB3000.dll")]
        public static extern int SetCtZEnable(int DevIndex, byte Chan, byte Enable);
        [DllImport("USB3000.dll")]
        public static extern int SetCtSoftTrig(int DevIndex, byte Chan);
        [DllImport("USB3000.dll")]
        public static extern int SetCtClrFifo(int DevIndex, byte Chan);

        //--------------------------------------------------------------------------
        // Trig Control

        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AiSoftTrig(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DiSoftTrig(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3DoSoftTrig(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3AoSoftTrig(int DevIndex, byte Chan);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3GlobalSoftTrig(int DevIndex);

        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrTrigger(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrAiTrigger(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrDiTrigger(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrDoTrigger(int DevIndex);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrAoTrigger(int DevIndex, byte Chan);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ClrGlobalSoftTrig(int DevIndex);

        //--------------------------------------------------------------------------
        // Sync Configuration

        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ExtTrigOutSource(int DevIndex, byte Source);
        [DllImport("USB3000.dll")]
        public static extern int SetUSB3ExtConvOutSource(int DevIndex, byte Source);

        //--------------------------------------------------------------------------
        // Get Data Acquired

        [DllImport("USB3000.dll")]
        public static extern int USB3GetAi(int DevIndex, uint Points, float[] Ai, int TimeOut);
        [DllImport("USB3000.dll")]
        public static extern int USB3GetDi(int DevIndex, uint Points, byte[] Di, int TimeOut);
        [DllImport("USB3000.dll")]
        public static extern int USB3GetCt(int DevIndex, byte Chan, uint Points, uint[] Ct, int TimeOut);
        [DllImport("USB3000.dll")]
        public static extern int GetLatestCtValue(int DevIndex, byte Chan, ref uint Ct, int TimeOut);
    }
}